'********************************************************************************
'
'                       C O P Y R I G H T  (c) 2002
'               N O R T H E R N    D Y N A M I C   I N C.
'                           All Rights Reserved.
'   ........................................................................
'   This sample code is provided by Software Toolbox solely to assist in
'   understanding the use of the SLIK-DA ActiveX Control. This code is
'   provided as-is and without warranty or support of any sort.
'
'********************************************************************************
'
'   Project:        SLIK-DA ActiveX Control
'
'   Description:    This sample server application is based on Software Toolbox's
'                   Simple Language Independent Toolkit for creating OPC
'                   Data Access servers (SLIK-DA). This implementation illustrates
'                   the use of various features of the SLIK-DA ActiveX Control.
'
'   Revision:
'       02-10-31    gde     Initial release.
'
'********************************************************************************

Option Strict Off
Option Explicit On

Imports System.Runtime.InteropServices

Module mdlSimulate
    Public Sub UpdateTags(ByRef Tags As NDI.SLIKDA.Interop.ISLIKTags)

        '
        ' A real server might scan one or more devices here to update the Tags.
        ' This server does not communicate with any hardware, so instead we will
        ' simulate data changes.
        '

        Dim Tag As NDI.SLIKDA.Interop.ISLIKTag
        Dim nIndex As Short

        ' Update each SLIKTag in the SLIKTags collection.
        For Each Tag In Tags
            '
            ' Simulate data changes for each tag that meets the following criteria:
            '   1. The tag is currently active.
            '   2. The tag's access permessions are read-only.
            '
            If Tag.Active And (Tag.AccessPermissions = NDI.SLIKDA.Interop.AccessPermissionsEnum.sdaReadAccess) Then
                Call SimulateDataChange(Tag, NDI.SLIKDA.Interop.QualityStatusEnum.sdaGood)
            Else
                ' Update the quality (and timestamp) for the other tags
                Call Tag.SetVQT(, NDI.SLIKDA.Interop.QualityStatusEnum.sdaGood)
            End If
        Next Tag

    End Sub

    Public Sub ReadTags(ByVal Count As Integer, ByRef Tags() As NDI.SLIKDA.Interop.ISLIKTag, ByRef Errors() As Integer, ByRef Result As Integer)

        '
        ' A real server might have to read one or more devices here to service the
        ' Read request.  This server does not communicate with any hardware.
        ' Instead, we rely on the periodic tag update to simulate data changes, so
        ' this Read call will be basically a no-op.

        Dim Tag As NDI.SLIKDA.Interop.ISLIKTag
        Dim nIndex As Short
        Dim sSrcPrefix As String

        ' Update the tag level error codes.
        For nIndex = 0 To Count - 1 ' Arrays from SLIK-DA are always 0-based.
            ' For this server, Tag reads always succeed!
            Errors(nIndex) = NDI.SLIKDA.Interop.OPCDAErrorsEnum.sdaSOK
        Next nIndex

        ' Return overall status.  For this server, the overall Read status is
        ' always sdaSOK.  A real server should set this according to the tag level
        ' read results.
        Result = NDI.SLIKDA.Interop.OPCDAErrorsEnum.sdaSOK

    End Sub

    Public Sub WriteTags( _
        ByVal Count As Integer, _
        ByRef Tags() As NDI.SLIKDA.Interop.ISLIKTag, _
        ByRef Values() As Object, _
        ByRef Qualities() As Short, _
        ByRef Timestamps() As System.DateTime, _
        ByRef Errors() As Integer, _
        ByRef Result As Integer _
    )
        ' Our simulation server does not talk to any hardware, so we just proceed to
        ' write the new values provided to the corresponding Tag.  A real server may
        ' validate the data first (e.g. range checking), scale the data, etc. before
        ' performing the actual write to the underlying system/hardware.

        Dim nIndex As Short

        Dim Quality As Short
        Quality = NDI.SLIKDA.Interop.QualityStatusEnum.sdaGood

        Dim Timestamp As System.DateTime
        Timestamp = NDI.SLIKDA.Interop.DefaultValues.SetVQT_Timestamp

        ' Write each tag
        For nIndex = 0 To Count - 1 ' Arrays from SLIK-DA are always 0-based.
            Dim Val As Object
            If TypeOf Values(nIndex) Is System.Decimal Then
                Val = New CurrencyWrapper(Values(nIndex))
            Else
                Val = Values(nIndex)
            End If

            If Qualities.Length > 0 Then
                Quality = Qualities(nIndex)
            End If
            If Timestamps.Length > 0 Then
                Timestamp = Timestamps(nIndex)
            End If

            Call Tags(nIndex).SetVQT(Val, Quality, Timestamp)
            Errors(nIndex) = NDI.SLIKDA.Interop.OPCDAErrorsEnum.sdaSOK
        Next nIndex

        ' Return overall status.  For this server, the overall Write status is
        ' always sdaSOK.  A real server should set this according to the tag level
        ' write results.
        Result = NDI.SLIKDA.Interop.OPCDAErrorsEnum.sdaSOK

    End Sub

    Private Sub SimulateDataChange(ByRef Tag As NDI.SLIKDA.Interop.ISLIKTag, ByRef nQuality As Short)

        Dim Value As Object
        Value = Tag.Value

        Select Case Tag.DataType
            Case VariantType.Boolean
                Value = Not Value
            Case VariantType.Byte
                If (Value > 250) Then
                    Value = 0
                Else
                    Value = Value + 1
                End If
            Case VariantType.Short
                If (Value > 10000) Then
                    Value = -10000
                Else
                    Value = Value + 4
                End If
            Case VariantType.Integer
                If (Value > 1000000) Then
                    Value = -1000000
                Else
                    Value = Value + 9
                End If
            Case VariantType.Single
                If (Value > 1.0E+20) Then
                    Value = -1.0E+20
                End If
                If (Value > 0) Then
                    Value = Value * 1.1
                Else
                    Value = (Value / 1.1) + 1
                End If
            Case VariantType.Double
                If (Value > 1.0E+50) Then
                    Value = -1.0E+50
                End If
                If (Value > 0) Then
                    Value = Value * 1.2
                Else
                    Value = (Value / 1.2) + 1
                End If
            Case VariantType.Currency
                Value = New CurrencyWrapper(Value + 1)
            Case VariantType.Date
                Value = Now
            Case VariantType.String
                Value = "Current time is " & CStr(Now)
            Case Else
                ' Simulation for data type not supported
        End Select

        ' Now set the Value, Quality, and Timestamp
        ' Note: By NOT passing the optional timestamp to SetVQT()
        '       SLIK-DA will use the current system time,
        '       and generate a timestamp with millisecond resolution.
        '       VB's Now() function, only gives 1 second
        '       resolution.  In a real server, the timestamp would
        '       more likely come from the underlying system.
        Call Tag.SetVQT(Value, nQuality)

    End Sub
End Module